<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Query;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;
use Cake\Cache\Cache;
use Cake\Utility\Text;
/**
 * Menus Model
 *
 * @property \App\Model\Table\PageContentsTable&\Cake\ORM\Association\HasMany $PageContents
 * @property \App\Model\Table\PageProductsTable&\Cake\ORM\Association\HasMany $PageProducts
 *
 * @method \App\Model\Entity\Menu newEmptyEntity()
 * @method \App\Model\Entity\Menu newEntity(array $data, array $options = [])
 * @method array<\App\Model\Entity\Menu> newEntities(array $data, array $options = [])
 * @method \App\Model\Entity\Menu get(mixed $primaryKey, array|string $finder = 'all', \Psr\SimpleCache\CacheInterface|string|null $cache = null, \Closure|string|null $cacheKey = null, mixed ...$args)
 * @method \App\Model\Entity\Menu findOrCreate($search, ?callable $callback = null, array $options = [])
 * @method \App\Model\Entity\Menu patchEntity(\Cake\Datasource\EntityInterface $entity, array $data, array $options = [])
 * @method array<\App\Model\Entity\Menu> patchEntities(iterable $entities, array $data, array $options = [])
 * @method \App\Model\Entity\Menu|false save(\Cake\Datasource\EntityInterface $entity, array $options = [])
 * @method \App\Model\Entity\Menu saveOrFail(\Cake\Datasource\EntityInterface $entity, array $options = [])
 * @method iterable<\App\Model\Entity\Menu>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\Menu>|false saveMany(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\Menu>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\Menu> saveManyOrFail(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\Menu>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\Menu>|false deleteMany(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\Menu>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\Menu> deleteManyOrFail(iterable $entities, array $options = [])
 */
class MenusTable extends Table
{
    /**
     * Initialize method
     *
     * @param array<string, mixed> $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('menus');
        $this->setDisplayField('name');
        $this->setPrimaryKey('id');

        $this->addBehavior('Timestamp');

        $this->hasMany('Albums', [
            'foreignKey' => 'menu_id',
        ]);
        $this->hasMany('Contents', [
            'foreignKey' => 'menu_id',
        ]);
        $this->hasMany('PageContents', [
            'foreignKey' => 'menu_id',
        ]);
        $this->hasMany('PageProducts', [
            'foreignKey' => 'menu_id',
        ]);
        $this->belongsTo('Menus', [
            'foreignKey' => 'parent_menu_id'
        ]);
        $this->belongsTo('ParentMenus', [
            'foreignKey' => 'parent_menu_id',
            'className' => 'Menus'
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator): Validator
    {
        $validator
            ->scalar('name')
            ->maxLength('name', 255)
            ->allowEmptyString('name');

        $validator
            ->scalar('description')
            ->maxLength('description', 255)
            ->allowEmptyString('description');

        $validator
            ->integer('parent_menu_id')
            ->notEmptyString('parent_menu_id');

        $validator
            ->integer('status')
            ->notEmptyString('status');

        $validator
            ->scalar('slug')
            ->maxLength('slug', 255)
            ->allowEmptyString('slug');

        $validator
            ->integer('show_on_home')
            ->allowEmptyString('show_on_home');

        $validator
            ->integer('show_on_footer')
            ->notEmptyString('show_on_footer');

        $validator
            ->scalar('image_path')
            ->maxLength('image_path', 255)
            ->allowEmptyString('image_path');

        $validator
            ->scalar('content')
            ->maxLength('content', 255)
            ->allowEmptyString('content');

        $validator
            ->integer('home_order')
            ->allowEmptyString('home_order');

        $validator
            ->integer('order_value')
            ->allowEmptyString('order_value');

        $validator
            ->scalar('url')
            ->maxLength('url', 255)
            ->allowEmptyString('url');

        $validator
            ->dateTime('date_created')
            ->allowEmptyDateTime('date_created');

        $validator
            ->dateTime('date_modified')
            ->notEmptyDateTime('date_modified');

        return $validator;
    }

    public function getMenuBySlug(string $slug)
    {
        return $this->find('all')
            ->where(['Menus.slug' => $slug])
            ->first();
    }

    public function getMainMenu()
    {
        return $this->find('all')
            ->where(['parent_menu_id' => 0, 'status' => 1])
            ->order(['order_value' => 'ASC'])
            ->toArray();
    }

    public function getSubmenusQuery(int $menuId)
    {
        return $this->find()
            ->where(['parent_menu_id' => $menuId])
            ->order(['order_value' => 'ASC']);
    }

    public function getSubmenus(int $menuId): array
    {
        return $this->getSubmenusQuery($menuId)->toArray();
    }

    public function getSubmenuOptions(int $menuId): array
    {
        return $this->getSubmenusQuery($menuId)
            ->select(['id', 'name'])
            ->all()
            ->combine('id', 'name')
            ->toArray();
    }

    public function getparentMenu($id)
    {
        $query = $this->find()
            ->where(['parent_menu_id' => $id])
            ->select(['id', 'name'])
            ->all()
            ->combine('id', 'name');

        return $query;
    }

    public function enaledMenus()
    {
        return $this->find('all')
            ->where(['status' => 1])
            ->order(['order_value' => 'ASC']);
    }

    public function parentMenuListing()
    {
        return $this->find('all')
            ->where(['parent_menu_id' => 0]);
    }

    public function pendingMenu()
    {
        return $this->find('all')->where(["status" => 0]);
    }

    public function enableMenu()
    {
        return $this->find('all')->where(["status" => 1]);
    }

    public function disableMenu()
    {
        return $this->find('all')->where(["status" => 99]);
    }

    public function editedMenu()
    {
        return $this->find('all')->where(["status" => 2]);
    }

    public function applySearchFilter(Query $query, $searchTerm)
    {
        if (!empty($searchTerm)) {
            $query->where([
                'OR' => [
                    'name LIKE' => '%' . $searchTerm . '%',
                    'content LIKE' => '%' . $searchTerm . '%',
                    'description LIKE' => '%' . $searchTerm . '%'
                ]
            ]);
        }

        return $query;
    }

    function findSubSubMenus($id = null)
    {
        $ids = [-1];

        $query = $this->find('all', ['conditions' => ['parent_menu_id' => $id]])->toArray();
        foreach ($query as $key => $q) {
            array_push($ids, $q['id']);
        }

        return $this->find('all')
            ->where(['parent_menu_id IN' => $ids]);
    }

    function findPendingSubSubMenus($id = null)
    {
        $ids = [-1];

        $query = $this->find('all', ['conditions' => ['parent_menu_id' => $id,]])->toArray();
        foreach ($query as $key => $q) {
            array_push($ids, $q['id']);
        }

        return $this->find('all')
            ->where(['parent_menu_id IN' => $ids, 'status' => 0]);
    }

    function findEnabledSubSubMenus($id = null)
    {
        $ids = [-1];

        $query = $this->find('all', ['conditions' => ['parent_menu_id' => $id,]])->toArray();
        foreach ($query as $key => $q) {
            array_push($ids, $q['id']);
        }

        return $this->find('all')
            ->where(['parent_menu_id IN' => $ids, 'status' => 1]);
    }

    function findDisabledSubSubMenus($id = null)
    {
        $ids = [-1];

        $query = $this->find('all', ['conditions' => ['parent_menu_id' => $id,]])->toArray();
        foreach ($query as $key => $q) {
            array_push($ids, $q['id']);
        }

        return $this->find('all')
            ->where(['parent_menu_id IN' => $ids, 'status' => 99]);
    }

    function galleryAsSubMenu()
    {
        return $this->find()
        ->where(['slug' =>'gallery','status'=> 1,'parent_menu_id !=' => 0])->first();
    }

}
