<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Query;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;
use Cake\Utility\Text;


/**
 * Contents Model
 *
 * @property \App\Model\Table\MenusTable&\Cake\ORM\Association\BelongsTo $Menus
 * @property \App\Model\Table\ContentImagesTable&\Cake\ORM\Association\HasMany $ContentImages
 * @property \App\Model\Table\PageContentsTable&\Cake\ORM\Association\HasMany $PageContents
 *
 * @method \App\Model\Entity\Content newEmptyEntity()
 * @method \App\Model\Entity\Content newEntity(array $data, array $options = [])
 * @method \App\Model\Entity\Content[] newEntities(array $data, array $options = [])
 * @method \App\Model\Entity\Content get($primaryKey, $options = [])
 * @method \App\Model\Entity\Content findOrCreate($search, ?callable $callback = null, $options = [])
 * @method \App\Model\Entity\Content patchEntity(\Cake\Datasource\EntityInterface $entity, array $data, array $options = [])
 * @method \App\Model\Entity\Content[] patchEntities(iterable $entities, array $data, array $options = [])
 * @method \App\Model\Entity\Content|false save(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \App\Model\Entity\Content saveOrFail(\Cake\Datasource\EntityInterface $entity, $options = [])
 * @method \App\Model\Entity\Content[]|\Cake\Datasource\ResultSetInterface|false saveMany(iterable $entities, $options = [])
 * @method \App\Model\Entity\Content[]|\Cake\Datasource\ResultSetInterface saveManyOrFail(iterable $entities, $options = [])
 * @method \App\Model\Entity\Content[]|\Cake\Datasource\ResultSetInterface|false deleteMany(iterable $entities, $options = [])
 * @method \App\Model\Entity\Content[]|\Cake\Datasource\ResultSetInterface deleteManyOrFail(iterable $entities, $options = [])
 *
 * @mixin \Cake\ORM\Behavior\TimestampBehavior
 */
class ContentsTable extends Table
{
    /**
     * Initialize method
     *
     * @param array $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('contents');
        $this->setDisplayField('name');
        $this->setPrimaryKey('id');

        $this->addBehavior('Timestamp');

        $this->belongsTo('Menus', [
            'foreignKey' => 'menu_id',
        ]);
        $this->hasMany('ContentImages', [
            'foreignKey' => 'content_id',
        ]);
        $this->hasMany('PageContents', [
            'foreignKey' => 'content_id',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator): Validator
    {
        $validator
            ->scalar('name')
            ->maxLength('name', 255)
            ->requirePresence('name', 'create')
            ->notEmptyString('name');

        $validator
            ->scalar('slug')
            ->maxLength('slug', 255)
            ->allowEmptyString('slug');

        $validator
            ->integer('menu_id')
            ->allowEmptyString('menu_id');

        $validator
            ->scalar('content')
            // ->maxLength('content', 16777215)
            // ->notEmptyString('content');
            ->allowEmptyString('content');


        $validator
            ->scalar('caption')
            ->maxLength('caption', 100)
            ->allowEmptyString('caption');

        $validator
            ->scalar('image_path')
            ->maxLength('image_path', 255)
            ->allowEmptyFile('image_path');

        $validator
            ->scalar('video_link')
            ->maxLength('video_link', 255)
            // ->requirePresence('video_link', 'create')
            ->allowEmptyString('video_link');

        $validator
            ->scalar('urls')
            ->maxLength('urls', 255)
            ->allowEmptyString('urls');

        // $validator
        //     ->scalar('file_path')
        //     ->maxLength('file_path', 255)
        //     ->allowEmptyString('file_path');

        $validator
            ->integer('status')
            ->notEmptyString('status');

        $validator
            ->scalar('level')
            ->maxLength('level', 255)
            ->allowEmptyString('level');

        $validator
            ->integer('show_on_home')
            ->allowEmptyString('show_on_home');

        $validator
            ->scalar('fa_icon')
            ->maxLength('fa_icon', 100)
            ->allowEmptyString('fa_icon');

        $validator
            ->integer('order_value')
            ->allowEmptyString('order_value');

        $validator
            ->scalar('info1')
            ->maxLength('info1', 255)
            ->allowEmptyString('info1');

        $validator
            ->scalar('map_address')
            ->maxLength('map_address', 500)
            ->allowEmptyString('map_address');

        $validator
            ->integer('show_on_footer')
            ->notEmptyString('show_on_footer');

        $validator
            ->dateTime('created_date')
            ->allowEmptyDateTime('created_date');

        $validator
            ->dateTime('modified_date')
            ->allowEmptyDateTime('modified_date');

        $validator
            ->integer('section_id')
            ->allowEmptyString('section_id');

        $validator
            ->integer('price')
            ->allowEmptyString('price');

        return $validator;
    }

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules): RulesChecker
    {
        $rules->add($rules->existsIn('menu_id', 'Menus'), ['errorField' => 'menu_id']);

        return $rules;
    }

    public function applySearchFilter(Query $query, $searchTerm)
    {
        if (!empty($searchTerm)) {
            $query->where([
                'OR' => [
                    'Contents.name LIKE' => '%' . $searchTerm . '%',
                    'Contents.content LIKE' => '%' . $searchTerm . '%',
                    'Contents.caption LIKE' => '%' . $searchTerm . '%'
                ]
            ]);
        }

        return $query;
    }



    public function parseHtmlToBlocksSmart($html)
    {
        libxml_use_internal_errors(true);
        $dom = new \DOMDocument();
        $dom->loadHTML('<?xml encoding="utf-8" ?>' . $html);

        $body = $dom->getElementsByTagName('body')->item(0);
        $blocks = [];

        $typeCounters = [
            'paragraph' => 0,
            'list' => 0,
        ];

        foreach ($body->childNodes as $node) {
            if ($node->nodeType !== XML_ELEMENT_NODE)
                continue;

            $type = '';
            $content = '';
            $block = [];
            $label = null;

            // Optional: look for preceding comment node to manually assign label
            $prev = $node->previousSibling;
            while ($prev && $prev->nodeType !== XML_COMMENT_NODE) {
                $prev = $prev->previousSibling;
            }

            if ($prev && $prev->nodeType === XML_COMMENT_NODE) {
                if (preg_match('/label\s*:\s*(\w+)/i', $prev->nodeValue, $matches)) {
                    $label = $matches[1];
                }
            }

            if ($node->nodeName === 'p') {
                $type = 'paragraph';
                $content = trim($node->textContent);

                if ($content !== '') {
                    $block = [
                        'type' => $type,
                        'label' => $label ?? "{$type}_" . $typeCounters[$type]++,
                        'content' => $content
                    ];
                }
            }

            if (in_array($node->nodeName, ['ul', 'ol'])) {
                $type = 'list';
                $items = [];
                foreach ($node->childNodes as $li) {
                    if ($li->nodeName === 'li') {
                        $items[] = trim($li->textContent);
                    }
                }

                $block = [
                    'type' => $type,
                    'label' => $label ?? "{$type}_" . $typeCounters[$type]++,
                    'style' => $node->nodeName === 'ul' ? 'unordered' : 'ordered',
                    'items' => $items
                ];
            }

            if (!empty($block)) {
                $blocks[] = $block;
            }
        }

        return json_encode($blocks);
    }



}
