<?php
declare(strict_types=1);

namespace App\Controller;

use App\Controller\AppController;
use cake\ORM\TableRegistry;
use Cake\Utility\Inflector;
use Cake\I18n\Time;
use Cake\Event\EventInterface;
use Cake\Core\Configure;
use Cake\Log\Log;


/**
 * Albums Controller
 *
 * @property \App\Model\Table\AlbumsTable $Albums
 * @method \App\Model\Entity\Album[]|\Cake\Datasource\ResultSetInterface paginate($object = null, array $settings = [])
 */
class AlbumsController extends AppController
{
    public function initialize(): void
    {
        parent::initialize();
        $this->albums = $this->fetchTable('Albums');
    }

    public function beforeRender(EventInterface $event)
    {
        parent::beforeRender($event);
    }

    public function index($action = null)
    {
        $pageTitle = "Album";
        $listings = $this->albums->findAlbums();
        $listings = $this->paginate($listings);
        $this->set(compact('pageTitle', 'listings'));
    }

    public function add($type = null)
    {
        // Create a new album entity
        $album = $this->albums->newEmptyEntity();
        $parentData = [];

        if ($this->request->is('post')) {

            $data = $this->request->getData();
            $image_path = $data['image_path'];
            $imageSize = $image_path->getSize();

            if ($imageSize > 0) {
                $fileOK = $this->uploadFiles(Configure::read('IMAGE_PREPATH'), $image_path, null, null, false);

                if (!empty($fileOK['urls'])) {
                    $data['image_path'] = substr(
                        $fileOK['file_path'][0],
                        Configure::read('IMAGE_PRELENGTH'),
                        strlen($fileOK['file_path'][0])
                    );
                }
            } else {
                $data['image_path'] = 'temp';
            }

            $album = $this->albums->patchEntity($album, $data);

            // Attempt to save the new album entity
            try {
                if ($this->albums->save($album)) {
                    $this->Flash->success(__('The album has been saved successfully.'));
                    Log::info('Album added successfully', ['album_id' => $album->id]);
                    return $this->redirect(['controller' => 'Gallery', 'action' => 'albums']);
                } else {
                    $this->Flash->error(__('The album could not be saved. Please, try again.'));
                    Log::warning('Album save failed', ['errors' => $album->getErrors()]);
                }
            } catch (\Exception $e) {
                $this->Flash->error(__('An error occurred while saving the album. Please, try again.'));
                Log::error('Exception occurred while saving album', ['exception' => $e]);
            }

        }

        // Set view variables
        $pageTitle = "Add album";
        $this->set(compact('album', 'pageTitle'));
    }

    public function edit($id = null)
    {

        // Retrieve the album entity by ID
        try {
            $album = $this->albums->get($id);
        } catch (\Exception $e) {
            $this->Flash->error(__('Album not found.'));
            Log::error('Failed to retrieve album', ['album_id' => $id, 'exception' => $e]);
            return $this->redirect(['action' => 'index']);
        }

        $parentData = [];

        if ($this->request->is(['post', 'patch', 'put'])) {
            $data = $this->request->getData();
            $image_path = $data['image_path'];

            if (isset($image_path)) {
                $fileOK = $this->uploadFiles(Configure::read('IMAGE_PREPATH'), $image_path, null, null, false);

                if (!empty($fileOK['urls'])) {
                    $data['image_path'] = substr(
                        $fileOK['file_path'][0],
                        Configure::read('IMAGE_PRELENGTH'),
                        strlen($fileOK['file_path'][0])
                    );
                }
            }

            $data['status'] = 2;
            $album = $this->albums->patchEntity($album, $data);

            // Attempt to save the updated album entity
            try {
                if ($this->albums->save($album)) {
                    $this->Flash->success(__('The album has been updated successfully.'));
                    Log::info('Album updated successfully', ['album_id' => $id]);
                    return $this->redirect(['controller' => 'Gallery', 'action' => 'albums']);
                } else {
                    $this->Flash->error(__('The album could not be updated. Please, try again.'));
                    Log::warning('Album update failed', ['album_id' => $id, 'errors' => $album->getErrors()]);
                }
            } catch (\Exception $e) {
                $this->Flash->error(__('An error occurred while updating the album. Please, try again.'));
                Log::error('Exception occurred while updating album', ['album_id' => $id, 'exception' => $e]);
            }

        }

        $pageTitle = 'Edit album';
        $this->set(compact('album', 'pageTitle', ));
    }

    public function enable($id)
    {
        $pageTitle = 'Enabled Images';
        $listings = $this->albums->enable($id);

        if ($listings) {
            $this->Flash->success(__('Album has been enabled successfully.'));
            Log::info('Album enabled', ['album_id' => $id]);
        } else {
            $this->Flash->error(__('Unable to enable the album. Please try again.'));
            Log::warning('Album enable failed, no rows affected', ['album_id' => $id]);
        }
        $this->set(compact('pageTitle', 'listings'));
        return $this->redirect($this->referer());
    }

    public function disable($id)
    {
        $pageTitle = 'Enabled Images';
        $listings = $this->albums->disable($id);
        // Check if any row was updated
        if ($listings) {
            $this->Flash->success(__('Album has been disabled successfully.'));
            Log::info('Album disabled', ['album_id' => $id]);
        } else {
            $this->Flash->error(__('Unable to disable the album. Please try again.'));
            Log::warning('Album disable failed, no rows affected', ['album_id' => $id]);
        }
        $this->set(compact('pageTitle', 'listings'));
        return $this->redirect($this->referer());
    }
}